/* 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 20015–2018 by Michael Hoffer
 * 
 * This file is part of SonoAir.
 *
 * SonoAir is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3
 * as published by the Free Software Foundation.
 * 
 * see: http://opensource.org/licenses/GPL-3.0
 *
 * VRL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * This version of SonoAir includes copyright notice and attribution requirements.
 * According to the GPL this information must be displayed even if you modify
 * the source code of SonoAir. Neither the SonoAir attribution icon(s) nor any
 * copyright statement/attribution may be removed.
 */
package eu.mihosoft.sonoair;

import com.sun.javafx.menu.MenuBase;
import com.sun.javafx.scene.control.GlobalMenuAdapter;
import com.sun.javafx.tk.Toolkit;
import eu.mihosoft.upnp.sonos.ZoneGroup;
import eu.mihosoft.upnp.sonos.ZonePlayer;
import eu.mihosoft.upnp.sonos.ZonePlayers;
import eu.mihosoft.vrl.base.VJarUtil;
import eu.mihosoft.vrl.base.VRL;
import eu.mihosoft.vrl.base.VSysUtil;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.logging.Level;
import java.util.logging.Logger;
import javafx.application.Application;
import javafx.application.Platform;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.StringProperty;
import javafx.event.ActionEvent;
import javafx.fxml.FXMLLoader;
import javafx.geometry.Insets;
import javafx.geometry.Orientation;
import javafx.geometry.Pos;
import javafx.scene.Node;
import javafx.scene.Parent;
import javafx.scene.Scene;
import javafx.scene.control.Alert;
import javafx.scene.control.Alert.AlertType;
import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.ListCell;
import javafx.scene.control.ListView;
import javafx.scene.control.Menu;
import javafx.scene.control.MenuBar;
import javafx.scene.control.MenuItem;
import javafx.scene.control.Slider;
import javafx.scene.control.SplitPane;
import javafx.scene.control.TextArea;
import javafx.scene.control.ToggleButton;
import javafx.scene.input.KeyCode;
import javafx.scene.input.KeyCodeCombination;
import javafx.scene.input.KeyCombination;
import javafx.scene.input.MouseButton;
import javafx.scene.layout.AnchorPane;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.Pane;
import javafx.scene.layout.Priority;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.VBox;
import javafx.scene.paint.Color;
import javafx.scene.text.Font;
import javafx.scene.web.WebView;
import javafx.stage.Modality;
import javafx.stage.Stage;
import javafx.stage.StageStyle;

/**
 * SonoAir application class.
 *
 *
 *
 * @author Michael Hoffer &lt;info@michaelhoffer.de&gt;
 */
public class SonoAir extends Application {

    /**
     * Version string for app title etc.
     */
    public static final String VERSION_TEXT = "1.0 (BETA 7.0)";
    /**
     * Version string for preference folder.
     */
    public static final String VERSION_CONFIG_PATH = "1.0-beta-7.0";

    /**
     * Configuration controller.
     */
    private ConfigController configController;

    /**
     * Base folder.
     */
    private File baseDir;

    /**
     * Device view (shows discovered devices as list)
     */
    private ListView<ZoneGroupWrapper> deviceView;

    /**
     * main app stage.
     */
    private Stage primaryStage;

    /**
     * Redirectable stream to display sout/serr in text view.
     */
    private RedirectableStream outStream;
    /**
     * Log window.
     */
    private TextArea outputArea;

    /**
     * Start/Stop Btn text property.
     */
    private StringProperty btnTextProperty;

    /**
     * App window title property.
     */
    private StringProperty windowTitleProperty;

    /**
     * Splitview divider property. Can be used to show/hide the log.
     */
    private DoubleProperty dividerProperty;

    /**
     * Zone players object for UPnP discovery.
     */
    private volatile ZonePlayers zonePlayers;
    /**
     * Device list (comma separated ips)
     */
    private volatile String devices = "";
    /**
     * Player groups used for generating the device list.
     */
    private volatile List<ZoneGroup> payerGroups;
    /**
     * Retry counter for scanning devices.
     */
    private volatile int retryCounter;

    @Override
    public void start(Stage primaryStage) {

        // init config folder
        VRL.init();

        // run discovery (zone group discovery with timeout done later)
        zonePlayers = ZonePlayers.discover();

        initConfigController();

        if (!configController.
                getDisableExperimentalWarningDialogProperty().get()) {
            Alert alert = new Alert(AlertType.INFORMATION);
            alert.setTitle("Warning - Experimental Build");
            alert.setHeaderText("This is a beta version!");
            alert.showAndWait();
        }

        // init ui & menus
        SplitPane root = initMainUI(primaryStage);
        initMenus();
        initSceneAndStage(root, primaryStage);
        // scan and run the native process
        rescanAndReRunAirUPNP();
    }

    private void initConfigController() {
        configController = new ConfigController();
        configController.getEnableDebugProperty().addListener((ov) -> {
            if (configController.getEnableDebugProperty().get()) {
                System.setErr(outStream);
            } else {
                System.setErr(RedirectableStream.ORIGINAL_SERR);
            }
        });
        configController.loadConfig();
        if (configController.getEnableDebugProperty().get()) {
            System.setErr(outStream);
        } else {
            System.setErr(RedirectableStream.ORIGINAL_SERR);
        }
    }

    /**
     * Initializes the main UI.
     *
     * @param primaryStage1 stage for main ui
     * @return split pane that contains the main ui
     */
    private SplitPane initMainUI(Stage primaryStage1) {
        outputArea = new TextArea();
        outputArea.setMinHeight(0);
        outputArea.getStyleClass().add("output");
        outputArea.setWrapText(true);
        outStream = new RedirectableStream(System.out, outputArea);
        outStream.setRedirectToStdOut(true);
        outStream.setRedirectToUi(true);
        deviceView = new ListView<>();
        deviceView.setCellFactory((ListView<ZoneGroupWrapper> param) -> {
            ListCell<ZoneGroupWrapper> cell = new ListCell<ZoneGroupWrapper>() {

                StackPane pane = new StackPane();
                Label label = new Label();
                Button editBtn = new Button("\uf013");

                {
                    label.getStyleClass().add("device-label");
                    pane.getChildren().addAll(label, editBtn);

                    setOnMouseClicked((me) -> {
                        if (me.getClickCount() == 2
                                && me.getButton() == MouseButton.PRIMARY
                                && !getText().trim().isEmpty()) {
                            openControlView(this, getItem().getGroup());
                        }
                    });

                    editBtn.setOnAction((ae) -> {
                        openControlView(this, getItem().getGroup());
                    });

                    editBtn.setFont(Font.loadFont(
                            SonoAir.class.getResource(
                                    "/font/fontawesome-webfont.ttf").
                                    toExternalForm(), 14));

                    StackPane.setAlignment(label, Pos.CENTER);
                    StackPane.setAlignment(editBtn, Pos.CENTER_RIGHT);
                }

                @Override
                protected void updateItem(ZoneGroupWrapper item, boolean empty) {
                    super.updateItem(item, empty);
                    setText(null);

                    if (empty || item == null || item.toString().isEmpty()) {
                        setGraphic(null);
                    } else {
                        setGraphic(pane);
                    }
                    if (item != null) {
                        label.setText(item.toString());
                        editBtn.setDisable(item.getGroup() == null);
                    }
                }
            };

            cell.setAlignment(Pos.CENTER);
            return cell;
        });

        Label label = new Label("Devices:");
        label.getStyleClass().add("title");
        label.setAlignment(Pos.CENTER);
        BorderPane.setAlignment(label, Pos.CENTER);
        Button donateButton = new Button();
        donateButton.getStyleClass().add("paypal-donate-btn");
        donateButton.setMinWidth(120);
        donateButton.setMinHeight(40);
        donateButton.setOnAction((e) -> showDonationPage());
        BorderPane mainUI = new BorderPane();
        StackPane p = new StackPane(donateButton);
        AnchorPane.setTopAnchor(p, 0.);
        AnchorPane.setBottomAnchor(p, 0.);
        AnchorPane aPane = new AnchorPane(label, p);
        StackPane.setAlignment(donateButton, Pos.CENTER);
        AnchorPane.setLeftAnchor(label, 0.);
        AnchorPane.setRightAnchor(label, 0.);
        mainUI.setTop(aPane);
        mainUI.getStyleClass().add("background");
        mainUI.setSnapToPixel(true);
        mainUI.setCenter(deviceView);
        Button btn = new Button();
        btnTextProperty = btn.textProperty();
        btn.getStyleClass().add("restart-btn");
        btn.setText("Stop SonoAir");
        btn.setOnAction((ActionEvent event) -> {
            if (!isAirsonosRunning()) {
                btnTextProperty.set("Stop SonoAir");
                updateDeviceView();
                primaryStage1.setTitle("SonoAir v" + VERSION_TEXT);
                rescanAndReRunAirUPNP();
            } else {
                btnTextProperty.set("Start SonoAir");
                stopAirSonos();
            }
        });
        Pane btnPane = new StackPane(btn);
        BorderPane.setAlignment(btnPane, Pos.CENTER);
        btnPane.getStyleClass().add("btn-pane");
        mainUI.setBottom(btnPane);
        mainUI.setMinHeight(215);
        SplitPane root = new SplitPane();
        root.getStyleClass().add("split-pane");
        root.setOrientation(Orientation.VERTICAL);
        root.getItems().addAll(mainUI, outputArea);
        root.setDividerPositions(1.0);
        dividerProperty = root.getDividers().get(0).positionProperty();
        return root;
    }

    /**
     * Initializes the scene and the specified stage.
     *
     * @param root root ui pane
     * @param primaryStage1 stage that shall be used for the specified ui
     */
    private void initSceneAndStage(SplitPane root, Stage primaryStage1) {
        Scene scene = new Scene(root, 650, 400);
        scene.getStylesheets().add("/eu/mihosoft/sonoair/maininterface.css");
        baseDir = VJarUtil.getClassLocation(SonoAir.class).getAbsoluteFile().
                getParentFile();
        this.primaryStage = primaryStage1;
        windowTitleProperty = primaryStage1.titleProperty();
        windowTitleProperty.set("SonoAir v" + VERSION_TEXT);
        primaryStage1.setScene(scene);
        primaryStage1.setResizable(false);
        primaryStage1.initStyle(StageStyle.UNIFIED);
        primaryStage1.setIconified(
                configController.getStartMinimizedProperty().get());
        primaryStage1.show();
    }

    /**
     * Initializes the menubar/menus.
     */
    private void initMenus() {
        MenuBar menuBar = new MenuBar();
        menuBar.setUseSystemMenuBar(true);

        Menu editMenu = new Menu("Edit");
        MenuItem prefItem = new MenuItem("Preferences");
        prefItem.setAccelerator(new KeyCodeCombination(
                KeyCode.COMMA, KeyCombination.META_DOWN));

        prefItem.setOnAction((e) -> {
            showPreferencewindow();
        });

        editMenu.getItems().add(prefItem);

        prefItem.setOnAction((e) -> {
            showPreferencewindow();
        });

        menuBar.getMenus().add(editMenu);

        Menu helpMenu = new Menu("Help");
        MenuItem aboutItem = new MenuItem("About SonoAir");

        aboutItem.setOnAction((e) -> {
            getHostServices().showDocument("http://sonoair.mihosoft.eu");
        });

        helpMenu.getItems().add(aboutItem);
        menuBar.getMenus().add(helpMenu);

        if (VSysUtil.isMacOSX()) {
            List<MenuBase> menus = new ArrayList<>();
            menus.add(GlobalMenuAdapter.adapt(editMenu));
            menus.add(GlobalMenuAdapter.adapt(helpMenu));
            Toolkit.getToolkit().getSystemMenu().setMenus(menus);
        }
    }

    /**
     * Rescans devices and runs airsonos module.
     */
    private void rescanAndReRunAirUPNP() {
        new Thread(() -> {
            stopAirSonos();

            if (configController.getEnableDebugProperty().get()) {
                outStream.println(">> DEBUG: retryCounter: " + retryCounter);
            }

            Platform.runLater(() -> btnTextProperty.set("Stop SonoAir"));

            scanDevices();

            payerGroups = zonePlayers.getDiscoveredZoneGroups();

            payerGroups = zonePlayers.getZoneGroups(
                    (int) configController.getDiscoveryTimeoutProperty().get());

            try {
                // sort entries
                Collections.sort(payerGroups,
                        (o1, o2) -> {
                            String name1 = o1.getName();
                            String name2 = o2.getName();

                            if (name1 == null || name1.isEmpty()) {
                                name1 = "< NO NAME >";

                                outStream.println(
                                        "DEBUG: Group with no name: "
                                        + o1.toString());
                            }

                            if (name2 == null || name2.isEmpty()) {
                                name2 = "< NO NAME >";

                                outStream.println(
                                        "DEBUG: Group with no name: "
                                        + o1.toString());
                            }

                            return name1.compareTo(name2);
                        });
            } catch (Exception ex) {
                ex.printStackTrace(System.err);
            }

            for (ZoneGroup player : payerGroups) {
                if (!devices.isEmpty()) {
                    devices += ",";
                }

                devices += player.getCoordinator().getIPv4Adress();
            }
            updateDeviceView();

            if ((devices != null && !devices.isEmpty())
                    || retryCounter >= 5) {
                runAirSonosInNewThread(devices);
                retryCounter = 0;
            } else {
                retryCounter++;
                rescanAndReRunAirUPNP();
            }
        }).start();
    }

    /**
     * Updates device view from discovered groups.
     */
    private void updateDeviceView() {
        deviceView.getItems().clear();
        for (ZoneGroup player : payerGroups) {
            addDeviceNotification(new ZoneGroupWrapper(player));
        }
    }

    /**
     * Opens the device control view for controlling volume and LED.
     *
     * @param n parent node (currently unused)
     * @param group zone group that shall be controlled
     */
    private void openControlView(Node n, ZoneGroup group) {

        String deviceName = group.getName();

        ZonePlayer zone = group.getCoordinator();

        BorderPane controlRoot = new BorderPane();
        controlRoot.getStyleClass().add("dev-control-background");

        Label label = new Label("Control " + deviceName);
        label.getStyleClass().add("device-title");
        label.setAlignment(Pos.CENTER);
        BorderPane.setAlignment(label, Pos.CENTER);
        controlRoot.setTop(label);

        Button doneBtn = new Button();
        doneBtn.getStyleClass().add("restart-btn");
        doneBtn.setText("Done");
        doneBtn.setAlignment(Pos.CENTER);
        doneBtn.setStyle("-fx-indent: 5 5 5 5;");
        doneBtn.setPrefHeight(20);
        BorderPane.setAlignment(doneBtn, Pos.CENTER);

        VBox controllerBox = new VBox();
        controllerBox.setPadding(new Insets(20, 0, 0, 0));

        controlRoot.setCenter(controllerBox);

        Slider volumeSlider = new Slider();

        volumeSlider.setValue(zone.getVolume());

        volumeSlider.setOnMouseReleased((me) -> {
            if (me.getButton() == MouseButton.PRIMARY) {
                zone.setVolume((int) volumeSlider.getValue());
            }
        });

        VBox.setVgrow(volumeSlider, Priority.ALWAYS);

        Label volumeLabel = new Label("Volume");
        volumeLabel.setAlignment(Pos.CENTER);
        volumeLabel.setMaxWidth(Double.MAX_VALUE);
        VBox.setVgrow(volumeLabel, Priority.ALWAYS);
        volumeLabel.getStyleClass().add("control-label");

        VBox sliderContainer = new VBox(volumeLabel, volumeSlider);

        controllerBox.getChildren().add(sliderContainer);

        ToggleButton ledToggleButton = new ToggleButton("");
        ledToggleButton.getStyleClass().add("led-btn");
        ledToggleButton.setMinHeight(20);
        ledToggleButton.setPrefHeight(20);
        ledToggleButton.setMaxHeight(20);
        ledToggleButton.setSelected(zone.isLedEnabled());

        ledToggleButton.selectedProperty().addListener((ov, oldV, newV) -> {
            zone.setLedEnabled(newV);
        });

        Label ledLabel = new Label("LED");
        ledLabel.getStyleClass().add("control-label");

        controllerBox.getChildren().add(new StackPane(ledLabel));
        controllerBox.getChildren().add(new StackPane(ledToggleButton));

        Stage stage = new Stage(StageStyle.TRANSPARENT);

        controlRoot.setBottom(doneBtn);
        doneBtn.setOnAction((ae) -> stage.close());

        Scene scene = new Scene(controlRoot, 400, 200);
        scene.setFill(Color.TRANSPARENT);
        scene.getStylesheets().add("/eu/mihosoft/sonoair/maininterface.css");

        stage.initOwner(primaryStage);
        stage.initModality(Modality.WINDOW_MODAL);
        stage.setTitle("Control Device: " + deviceName);
        stage.setScene(scene);
        stage.setResizable(false);
        stage.setX(primaryStage.getX() + primaryStage.getWidth() / 2 - 400);
        stage.setY(primaryStage.getY() + primaryStage.getHeight() / 2 - 200);
        stage.setOpacity(0.98);
        stage.show();

        stage.setX(primaryStage.getX() + primaryStage.getWidth() / 2 - stage.getWidth() / 2);
        stage.setY(primaryStage.getY() + primaryStage.getHeight() / 2 - stage.getHeight() / 2);
    }

    /**
     * Shows the preference view in its own stage.
     */
    private void showPreferencewindow() {

        FXMLLoader fxmlLoader = new FXMLLoader(
                getClass().
                        getResource("Preferences.fxml"));

        try {
            fxmlLoader.load();
        } catch (IOException ex) {
            Logger.getLogger(SonoAir.class.getName()).
                    log(Level.SEVERE, null, ex);
        }

        PreferencesController controller = fxmlLoader.getController();
        controller.setConfigController(configController);
        configController.loadConfig();

        Scene scene = new Scene((Parent) fxmlLoader.getRoot(), 800, 500);

        Stage configStage = new Stage();
        configStage.setScene(scene);
        configStage.setResizable(false);
        configStage.setTitle("SonoAir: Preferences");

        configStage.setOnCloseRequest((we) -> {
            if (configController.getEnableDebugProperty().get()) {
                System.setErr(outStream);
            } else {
                System.setErr(RedirectableStream.ORIGINAL_SERR);
            }
            rescanAndReRunAirUPNP();
        });

        configStage.show();

    }

    /**
     * Shows the donation page in its own stage.
     */
    private void showDonationPage() {
        Stage donationStage = new Stage(StageStyle.UNIFIED);

        WebView view = new WebView();
        view.getEngine().load("https://www.paypal.com/cgi-bin/webscr?cmd="
                + "_s-xclick&hosted_button_id=WQC8SVJ2P7PNG");

        Scene scene = new Scene(view, 800, 600);

        donationStage.setScene(scene);
        donationStage.setTitle("Donate");
        donationStage.initOwner(primaryStage);
        donationStage.initModality(Modality.WINDOW_MODAL);
        donationStage.show();
    }

    /**
     * Runs airsonos in a new thread.
     *
     * @param devices devie list (leave empty for airsonos fallback discovery)
     */
    private void runAirSonosInNewThread(final String devices) {
        Thread t = new Thread(() -> {
            runAirSonos(devices);
        });
        t.start();
    }

    @Override
    public void stop() throws Exception {
        stopAirSonos();
        System.exit(0);
    }

    /**
     * Indicates whether sonoair is running.
     *
     * @return {@code true} if sonoair is running; {@code false} otherwise
     */
    private boolean isAirsonosRunning() {
        return process != null;
    }

    /**
     * Runs AirUPNP.
     *
     * @param devices devie list (leave empty for airsonos fallback discovery)
     */
    private void runAirSonos(String devices) {

        final String airUPNP = "airconnect";

        dividerProperty.set(1);
        if (!configController.getEnableDebugProperty().get()) {
            outputArea.setText("");
        }
        outStream.println("SonoAir v" + VERSION_TEXT);
        outStream.println("Copyright (c) 2018 Michael Hoffer <info@michaelhoffer.de>");
        outStream.println("--------------------------------------------------------------");
        outStream.println("-> using AirUPNP");
        outStream.println("-> using AirUPNP by philippe_44@outlook.com");
        outStream.println("   see https://github.com/philippe44/AirConnect/blob/master/LICENSE");
        outStream.println("-> the source code of SonoAir/AirUPNP is contained within the app bundle");
        outStream.println("   (see src folder)");
        outStream.print("-> starting AirUPNP module...");

        if (isAirsonosRunning()) {
            stopAirSonos();
        }

        List<String> execArgs = new ArrayList<>();

        execArgs.add(baseDir.getPath() + "/resources/osx/" + airUPNP + "/bin/airupnp-osx-multi");
        execArgs.addAll(Arrays.asList("-l", "1000:2000", "-z"));

        if (devices != null && !devices.trim().isEmpty()) {
            Platform.runLater(() -> primaryStage.setTitle("SonoAir v" + VERSION_TEXT));
            if (configController.getEnableDebugProperty().get()) {
                // outStream.println("\nWARNING: devices ignored (TODO 05.11.2015)");
            }
//            execArgs.add("--device-list");
//            execArgs.add(devices);
        }

        ProcessBuilder builder
                = new ProcessBuilder(
                        execArgs.toArray(new String[execArgs.size()]));
        builder.redirectErrorStream(true);
        try {

            ProcessBuilder execBuilder = new ProcessBuilder(
                    "chmod", "a+x",
                    baseDir.getPath() + "/resources/osx/" + airUPNP + "/bin/airupnp-osx-multi");
            builder.redirectErrorStream(true);

            Process makeExecutable = execBuilder.start();

            BufferedReader executableReader = new BufferedReader(
                    new InputStreamReader(makeExecutable.getInputStream()));

            String execLine;

            while ((execLine = executableReader.readLine()) != null) {
                //
            }

            process = builder.start();

            outStream.println(" [done]\n--------------------------------------------------------------");

            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(process.getInputStream()));

            String line;

            while ((line = reader.readLine()) != null) {

                if (line.contains("adding renderer")) {

                    String addingPlayerStr
                            = line.substring(line.lastIndexOf("adding renderer")).
                                    replace("adding renderer",
                                            "-> device found");

                    outStream.println(addingPlayerStr);

                } else {
                    // outStream.println(line);
                }
            }

        } catch (IOException ex) {
            Logger.getLogger(SonoAir.class.getName()).
                    log(Level.SEVERE, null, ex);
            outStream.println(" [failed]");
            dividerProperty.set(0);
            stopAirSonos();
        }
    }

    /**
     * Adds device to device view.
     *
     * @param group group to add
     */
    private void addDeviceNotification(ZoneGroupWrapper group) {
        System.out.println("Adding device " + group.toString());
        Platform.runLater(() -> {
            deviceView.getItems().add(group);
            primaryStage.setTitle("SonoAir v" + VERSION_TEXT);
        });
    }

    /**
     * Changes the app title to "searching devices...".
     */
    private void scanDevices() {
        Platform.runLater(() -> {
            primaryStage.setTitle("SonoAir v" + VERSION_TEXT + " (searching devices...)");
        });
    }

    /**
     * Fallback method for backward compatibility.
     *
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        launch(args);
    }

    /**
     * Airsonos process.
     */
    private Process process;

    /**
     * Stops a running airsonos process.
     */
    private void stopAirSonos() {

        devices = "";

        Platform.runLater(() -> {
            windowTitleProperty.set("SonoAir v" + VERSION_TEXT);
            deviceView.getItems().clear();
            btnTextProperty.set("Start SonoAir");
        });

        if (process != null) {

            try {
                process.destroyForcibly();
            } finally {
                process = null;
                outStream.println("--------------------------------------------------------------");
                outStream.println("-> AirConnect module stopped.");
            }
        }
    }

    public static void runAndWait(Runnable action) {
        if (action == null) {
            throw new NullPointerException("action");
        }

        // run synchronously on JavaFX thread
        if (Platform.isFxApplicationThread()) {
            action.run();
            return;
        }

        // queue on JavaFX thread and wait for completion
        final CountDownLatch doneLatch = new CountDownLatch(1);
        Platform.runLater(() -> {
            try {
                action.run();
            } finally {
                doneLatch.countDown();
            }
        });

        try {
            doneLatch.await();
        } catch (InterruptedException e) {
            // ignore exception
        }
    }

}
